<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

if ( ! function_exists( 'cbio_is_licence_valid_and_active' ) ) {
	require_once plugin_dir_path( __FILE__ ) . 'settings-page.php';
}

add_action('admin_footer-upload.php', 'cbio_add_bio_toolbar');

function cbio_add_bio_toolbar() {
	$is_license_active = cbio_is_licence_valid_and_active();
	$show_bulk_toolbar = function_exists('cbio_should_show_bulk_toolbar') ? cbio_should_show_bulk_toolbar() : true;

	if (!$is_license_active || !$show_bulk_toolbar) {
		return;
	}

	?>

	<script type="text/javascript">
		(function($) {
			$(document).ready(function() {
				var customToolbar = $('<div class="cbio-bulk-toolbar"></div>');

				var title = $('<span class="cbio-bulk-toolbar-title"><?php esc_html_e('Bulk Image Optimization', 'coding-bunny-image-optimizer'); ?></span>');
				customToolbar.append(title);

				var convertToWebpButton = $('<button class="button cbio-optimize-button"><?php esc_html_e('Optimize Selected Images', 'coding-bunny-image-optimizer'); ?></button>');
				convertToWebpButton.attr("id", "convert_to_webp_delete");

				var convertAllToWebpButton = $('<button class="button cbio-optimize-all-button"><?php esc_html_e('Optimize All Images', 'coding-bunny-image-optimizer'); ?></button>');
				convertAllToWebpButton.attr("id", "convert_all_to_webp_delete");

				customToolbar.append(convertToWebpButton);
				customToolbar.append(convertAllToWebpButton);

				$('.tablenav.top').after(customToolbar);

				function optimizeImages(ids, action) {
    var totalImages = ids.length;
    var optimizedImages = 0;
    var batchSize = <?php echo esc_js(get_option('cbio_batch_size', '5')); ?>; // Usa il valore dalle impostazioni
    
    for(var i = 0; i < ids.length; i += batchSize) {
						var batch = ids.slice(i, i + batchSize);
						
						(function(currentBatch, batchIndex) {
							setTimeout(function() {
								$.ajax({
									url: "<?php echo esc_url(admin_url('admin-ajax.php')); ?>",
									type: 'POST',
									data: {
										action: action,
										nonce: "<?php echo esc_js(wp_create_nonce('convert_webp_nonce')); ?>",
										ids: currentBatch,
										action_choice: 'delete'
									},
									success: function(response) {
										optimizedImages += currentBatch.length;
										
										if (optimizedImages >= totalImages) {
											alert('Optimization completed.');
											location.reload();
										}
									},
									error: function(xhr, status, error) {
										console.error("Error optimizing batch:", error);
										console.log("Server response:", xhr.responseText);
										alert('An error occurred while optimizing. Please check the console for details.');
									}
								});
							}, batchIndex * 500);
						})(batch, i / batchSize);
					}
				}

				$('#convert_to_webp_delete').on('click', function() {
					var selectedIds = [];
					$('input[type="checkbox"]:checked').each(function() {
						var id = $(this).val();
						if (id && id !== 'on') {
							selectedIds.push(id);
						}
					});

					if (selectedIds.length === 0) {
						alert('Please select at least one image.');
						return;
					}

					if (confirm('With optimization you are about to permanently delete these images from your site. This action cannot be undone. "Cancel" to stop, "OK" to delete.')) {
						const button = $(this);
						button.prop("disabled", true).text("Optimizing...");
						optimizeImages(selectedIds, 'convert_to_webp_multiple');
					}
				});

				$('#convert_all_to_webp_delete').on('click', function() {
					if (confirm('With optimization you are about to permanently delete these images from your site. This action cannot be undone. "Cancel" to stop, "OK" to delete.')) {
						const button = $(this);
						button.prop("disabled", true).text("Optimizing...");
						
						function processImageBatch(offset = 0) {
							$.ajax({
								url: "<?php echo esc_url(admin_url('admin-ajax.php')); ?>",
								type: 'POST',
								data: {
            action: 'convert_all_to_webp',
            nonce: "<?php echo esc_js(wp_create_nonce('convert_webp_nonce')); ?>",
            offset: offset
        },
								success: function(response) {
									if (response.success) {
										var data = response.data;
										
										if (!data.isDone) {
											processImageBatch(data.nextOffset);
										} else {
											button.prop("disabled", false).text("<?php esc_html_e('Optimize All Images', 'coding-bunny-image-optimizer'); ?>");
											alert('All images optimized.');
											location.reload();
										}
									}
								},
								error: function(xhr, status, error) {
									console.error("Error in batch processing:", error);
									console.log("Server response:", xhr.responseText);
									alert('An error occurred. Please check the console for details.');
									button.prop("disabled", false).text("<?php esc_html_e('Optimize All Images', 'coding-bunny-image-optimizer'); ?>");
								}
							});
						}
						
						processImageBatch();
					}
				});
			});
		})(jQuery);
	</script>

	<style type="text/css">
		.cbio-bulk-toolbar {
			width: max-content;
			margin-bottom: 10px;
			padding: 10px;
			background: white;
			border: 1px solid #C3C4C7;
			display: flex;
			align-items: center;
		}
		.cbio-bulk-toolbar-title {
			font-weight: 700;
			margin-right: 10px;
		}
		.cbio-optimize-button {
			margin-right: 10px !important;
			background-color: #7F54B2 !important;
			border-color: #7F54B2 !important;
			color: #FFFFFF !important;
		}
		.cbio-optimize-button:hover {
			background-color: #A98ED6 !important;
			border-color: #A98ED6 !important;
		}
		.cbio-optimize-all-button {
			margin-right: 10px !important;
			border-color: #7F54B2 !important;
			color: #7F54B2 !important;
		}
		.cbio-optimize-all-button:hover {
			border-color: #7F54B2 !important;
			color: #7F54B2 !important;
		}
	</style>

	<?php
}

add_action('wp_ajax_get_all_image_ids', 'cbio_get_all_image_ids');

function cbio_get_all_image_ids() {
	check_ajax_referer('convert_webp_nonce', 'nonce');

	if (!current_user_can('manage_options')) {
		wp_send_json_error(__('Unauthorized user', 'coding-bunny-image-optimizer'));
	}

	$cache_key = 'all_image_ids';
	$image_ids = wp_cache_get($cache_key, 'custom_cache_group');

	if ($image_ids === false) {
		$args = array(
			'post_type' => 'attachment',
			'post_mime_type' => 'image',
			'post_status' => 'inherit',
			'posts_per_page' => -1,
			'fields' => 'ids'
		);

		$query = new WP_Query($args);
		if ($query->have_posts()) {
			$image_ids = $query->posts;
			wp_cache_set($cache_key, $image_ids, 'custom_cache_group', 3600);
		} else {
			wp_send_json_error(__('No images found', 'coding-bunny-image-optimizer'));
		}
	}

	wp_send_json_success($image_ids);
}

add_action('wp_ajax_convert_to_webp_multiple', 'cbio_convert_to_webp_multiple');

function cbio_convert_to_webp_multiple() {
	check_ajax_referer('convert_webp_nonce', 'nonce');

	if (!current_user_can('manage_options')) {
		wp_send_json_error(__('Unauthorized user', 'coding-bunny-image-optimizer'));
	}

	if (!isset($_POST['ids']) || !is_array($_POST['ids']) || empty($_POST['ids'])) {
		wp_send_json_error(__('No images selected', 'coding-bunny-image-optimizer'));
	}

	$ids = array_map('intval', $_POST['ids']);
	$action_choice = sanitize_text_field($_POST['action_choice']);
	$convert_format = get_option('cbio_convert_format', 'webp');
	$delete_original = get_option('cbio_delete_original', '0');
	
	$results = array(
		'success' => 0,
		'failed' => 0,
		'skipped' => 0,
		'details' => array()
	);

	foreach ($ids as $post_id) {
		$result = optimize_image($post_id, $convert_format, $action_choice, $delete_original);
		if ($result === true) {
			$results['success']++;
			$results['details'][$post_id] = 'success';
		} elseif ($result === 'skipped') {
			$results['skipped']++;
			$results['details'][$post_id] = 'skipped';
		} else {
			$results['failed']++;
			$results['details'][$post_id] = 'failed';
		}
	}

	wp_send_json_success($results);
}

add_action('wp_ajax_convert_all_to_webp', 'cbio_convert_all_to_webp');

function cbio_convert_all_to_webp() {
    check_ajax_referer('convert_webp_nonce', 'nonce');

    if (!current_user_can('manage_options')) {
        wp_send_json_error(__('Unauthorized user', 'coding-bunny-image-optimizer'));
    }
    
    $batch_size = get_option('cbio_batch_size_all', 10);
    $offset = isset($_POST['offset']) ? intval($_POST['offset']) : 0;
	
	$args = array(
		'post_type' => 'attachment',
		'post_mime_type' => 'image',
		'post_status' => 'inherit',
		'posts_per_page' => $batch_size,
		'offset' => $offset,
		'fields' => 'ids'
	);
	
	$query = new WP_Query($args);
	$total_images = $query->found_posts;
	
	$results = array(
		'success' => 0,
		'failed' => 0,
		'skipped' => 0,
		'details' => array()
	);
	
	if ($query->have_posts()) {
		$batch_ids = $query->posts;
		$convert_format = get_option('cbio_convert_format', 'webp');
		$action_choice = 'delete';
		$delete_original = get_option('cbio_delete_original', '0');
		
		foreach ($batch_ids as $post_id) {
			$result = optimize_image($post_id, $convert_format, $action_choice, $delete_original);
			if ($result === true) {
				$results['success']++;
				$results['details'][$post_id] = 'success';
			} elseif ($result === 'skipped') {
				$results['skipped']++;
				$results['details'][$post_id] = 'skipped';
			} else {
				$results['failed']++;
				$results['details'][$post_id] = 'failed';
			}
		}
		
		wp_send_json_success(array(
			'processed' => $offset + count($batch_ids),
			'total' => $total_images,
			'nextOffset' => $offset + $batch_size,
			'isDone' => ($offset + $batch_size >= $total_images),
			'results' => $results
		));
	} else {
		wp_send_json_success(array(
			'isDone' => true,
			'message' => __('All images optimized.', 'coding-bunny-image-optimizer'),
			'results' => $results
		));
	}
}

function optimize_image($post_id, $convert_format, $action_choice, $delete_original = '0') {
	try {

		$is_optimized = get_post_meta($post_id, '_cbio_optimized', true);
		if ($is_optimized === $convert_format) {
			return 'skipped';
		}
		
		$image_path = get_attached_file($post_id);

		if (!file_exists($image_path)) {
			return false;
		}

		$image_size = filesize($image_path);
		$memory_limit = ini_get('memory_limit');
		$memory_limit_bytes = wp_convert_hr_to_bytes($memory_limit);
		$memory_required = $image_size * 5;
		
		if (memory_get_usage() + $memory_required > $memory_limit_bytes) {
			return false;
		}

		$info = pathinfo($image_path);
		$extension = isset($info['extension']) ? strtolower($info['extension']) : '';

		if (!in_array($extension, ['jpg', 'jpeg', 'png', 'gif'])) {
			return 'skipped';
		}

		$output_path = $info['dirname'] . '/' . $info['filename'] . '.' . $convert_format;
		$mime_type = 'image/' . $convert_format;

		if (file_exists($output_path)) {

			update_attached_file($post_id, $output_path);
			$metadata = wp_generate_attachment_metadata($post_id, $output_path);
			$attachment = array(
				'ID' => $post_id,
				'post_mime_type' => $mime_type
			);
			wp_update_post($attachment);
			wp_update_attachment_metadata($post_id, $metadata);
			
			update_post_meta($post_id, '_cbio_optimized', $convert_format);
			update_post_meta($post_id, '_cbio_optimized_date', current_time('mysql'));
			
			return true;
		}

		if (convert_image($image_path, $output_path, $convert_format)) {
			update_attached_file($post_id, $output_path);

			$metadata = wp_generate_attachment_metadata($post_id, $output_path);
			$attachment = array(
				'ID' => $post_id,
				'post_mime_type' => $mime_type
			);
			wp_update_post($attachment);
			wp_update_attachment_metadata($post_id, $metadata);
			
			update_post_meta($post_id, '_cbio_optimized', $convert_format);
			update_post_meta($post_id, '_cbio_optimized_date', current_time('mysql'));
			
			if ($delete_original === '1') {
				wp_delete_file($image_path);

				foreach (['jpg', 'jpeg', 'png', 'gif'] as $ext) {
					$old_file_path = $info['dirname'] . '/' . $info['filename'] . '.' . $ext;
					if (file_exists($old_file_path) && $old_file_path !== $output_path) {
						wp_delete_file($old_file_path);
					}

					$intermediate_sizes = glob($info['dirname'] . '/' . $info['filename'] . '-*.' . $ext);
					foreach ($intermediate_sizes as $intermediate_file) {
						wp_delete_file($intermediate_file);
					}
				}

				global $wpdb;
				
				$cache_key = 'post_content_replace_' . md5($info['basename'] . $info['filename'] . '.' . $convert_format);

				$cached_result = wp_cache_get($cache_key);

				if ($cached_result === false) {
					$result = $wpdb->query(
						$wpdb->prepare(
							"UPDATE {$wpdb->posts} SET post_content = REPLACE(post_content, %s, %s) WHERE post_content LIKE %s",
							$info['basename'],
							$info['filename'] . '.' . $convert_format,
							'%' . $info['basename'] . '%'
						)
					);

					wp_cache_set($cache_key, $result);
				} else {
					$result = $cached_result;
				}
			}
			
			return true;
		}
		
		return false;
	} catch (Exception $e) {
		return false;
	}
}

function convert_image($source, $destination, $format) {
	try {
		$max_width = get_option('cbio_max_width', 1000);
		$max_height = get_option('cbio_max_height', 1000);
		$enable_resize = get_option('cbio_enable_resize', '1');
		$enable_sharpening = get_option('cbio_enable_sharpening', '0');
		$sharpen_amount = get_option('cbio_sharpen_amount', 0.5);

		$quality_webp = get_option('cbio_quality_webp', 80);
		$quality_avif = get_option('cbio_quality_avif', 80);
		
		$image_info = getimagesize($source);
		$is_transparent = false;
		
		if ($image_info[2] === IMAGETYPE_PNG) {
			$quality_webp_png = get_option('cbio_quality_webp_png', 85);
			$quality_avif_png = get_option('cbio_quality_avif_png', 65);
			
			$img = @imagecreatefrompng($source);
			if ($img !== false) {
				for ($x = 0; $x < imagesx($img); $x++) {
					for ($y = 0; $y < imagesy($img); $y++) {
						$color = imagecolorat($img, $x, $y);
						$alpha = ($color >> 24) & 0x7F;
						if ($alpha > 0) {
							$is_transparent = true;
							break 2;
						}
					}
				}
				imagedestroy($img);
			}
		}

		if (extension_loaded('imagick')) {
			$imagick = new Imagick($source);
			
			$profiles = $imagick->getImageProfiles("icc", true);
			
			$width = $imagick->getImageWidth();
			$height = $imagick->getImageHeight();

			if ($enable_resize === '1' && ($width > $max_width || $height > $max_height)) {
				$imagick->resizeImage($max_width, $max_height, Imagick::FILTER_LANCZOS, 1, true);
				
				if ($enable_sharpening === '1') {
					$imagick->sharpenImage(0, $sharpen_amount);
				}
			}

			if ($format === 'webp') {

				$imagick->setOption('webp:method', '6');
				$imagick->setOption('webp:low-memory', 'true');
				$imagick->setOption('webp:alpha-compression', '1');
				
				if ($is_transparent) {
					$imagick->setOption('webp:alpha-quality', '100');
					$imagick->setImageCompressionQuality($quality_webp_png);
				} else {
					$imagick->setOption('webp:lossless', 'false');
					$imagick->setImageCompressionQuality($quality_webp);
				}
			} elseif ($format === 'avif') {
				if ($is_transparent) {
					$imagick->setImageCompressionQuality($quality_avif_png);
				} else {
					$imagick->setImageCompressionQuality($quality_avif);
				}

				$imagick->setOption('avif:threads', '4');
			}

			$imagick->setImageFormat($format);
			
			if (!empty($profiles)) {
				$imagick->profileImage("icc", $profiles['icc']);
			}
			
			$result = $imagick->writeImage($destination);
			$imagick->clear();
			$imagick->destroy();
			return $result;
			
		} elseif (function_exists('imagecreatefromjpeg') || function_exists('imagecreatefrompng') || function_exists('imagecreatefromgif')) {

			$image = null;
			$info = getimagesize($source);
			
			switch ($info[2]) {
				case IMAGETYPE_JPEG:
					$image = imagecreatefromjpeg($source);
					break;
				case IMAGETYPE_PNG:
					$image = imagecreatefrompng($source);
			
					if ($is_transparent) {
						imagealphablending($image, false);
						imagesavealpha($image, true);
					}
					break;
				case IMAGETYPE_GIF:
					$image = imagecreatefromgif($source);
					break;
			}
			
			if (!$image) {
				return false;
			}

			$width = imagesx($image);
			$height = imagesy($image);

			if ($enable_resize === '1' && ($width > $max_width || $height > $max_height)) {
				$aspect_ratio = $width / $height;
				if ($max_width / $max_height > $aspect_ratio) {
					$new_width = min($width, $max_height * $aspect_ratio);
					$new_height = min($height, $max_height);
				} else {
					$new_width = min($width, $max_width);
					$new_height = min($height, $max_width / $aspect_ratio);
				}
				
				$resized_image = imagecreatetruecolor($new_width, $new_height);
				
				if ($is_transparent) {
					imagealphablending($resized_image, false);
					imagesavealpha($resized_image, true);
					$transparent = imagecolorallocatealpha($resized_image, 255, 255, 255, 127);
					imagefilledrectangle($resized_image, 0, 0, $new_width, $new_height, $transparent);
				}
				
				imagecopyresampled($resized_image, $image, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
				imagedestroy($image);
				$image = $resized_image;
				
				if ($enable_sharpening === '1') {
					$sharpen_matrix = array(
						array(-1, -1, -1),
						array(-1, 16, -1),
						array(-1, -1, -1)
					);
					$divisor = array_sum(array_map('array_sum', $sharpen_matrix));
					$offset = 0;
					imageconvolution($image, $sharpen_matrix, $divisor, $offset);
				}
			}
			
			if (!imageistruecolor($image)) {
				imagepalettetotruecolor($image);
			}
			
			$result = false;
			if ($format === 'webp') {

				$quality = $is_transparent ? $quality_webp_png : $quality_webp;
				$result = imagewebp($image, $destination, $quality);
			} elseif ($format === 'avif' && function_exists('imageavif')) {
				$quality = $is_transparent ? $quality_avif_png : $quality_avif;
				$result = imageavif($image, $destination, $quality);
			}
			
			imagedestroy($image);
			return $result;
		}
		
		return false;
	} catch (Exception $e) {
		return false;
	}
}