<?php

if (!defined('ABSPATH')) {
	exit;
}

include_once plugin_dir_path(__DIR__) . '/includes/watermark-functions.php';

class Cbio_Watermark {
	private $options;

	public function __construct() {
		add_action('admin_menu', array($this, 'add_settings_page'), 10);
		add_action('admin_init', array($this, 'register_settings'));
		add_action('admin_enqueue_scripts', array($this, 'enqueue_scripts'));
		add_filter('wp_generate_attachment_metadata', array($this, 'apply_watermark'), 10, 2);
		add_filter('bulk_actions-upload', array($this, 'register_bulk_actions'));
		add_filter('handle_bulk_actions-upload', array($this, 'handle_bulk_actions'), 10, 3);
	}

	public function add_settings_page() {
		if ($this->is_license_active()) {
			add_submenu_page(
			'coding-bunny-image-optimizer',
			esc_html__('CodingBunny Image Watermark', 'coding-bunny-image-optimizer'),
			esc_html__('Watermark Settings', 'coding-bunny-image-optimizer'),
			'manage_options',
			'coding-bunny-image-watermark',
			array($this, 'create_settings_page')
		);
	}
}

public function is_license_active() {
	include_once plugin_dir_path(__DIR__) . '/admin/licence-validation.php';
	return cbio_is_licence_valid_and_active();
}

private function get_cached_watermark($image_width, $opacity, $size) {
	return Cbio_Watermark_Functions::get_cached_watermark($image_width, $opacity, $size);
}

public function create_settings_page() {
	$this->options = get_option('watermark_options');
	?>
	<div class="wrap">
		<h1><?php esc_html_e('CodingBunny Image Optimizer', 'coding-bunny-image-optimizer'); ?> 
			<span>v<?php echo esc_html(CBIO_VERSION); ?></span></h1>
			<div class="cbio-content">
				<div class="cbio-section">
					<h3><?php esc_html_e('Watermark Settings', 'coding-bunny-image-optimizer'); ?></h3>
					<p><?php esc_html_e('Configure your watermark settings below.', 'coding-bunny-image-optimizer'); ?></p>
					<form method="post" action="options.php">
						<?php
						settings_fields('watermark_option_group');
						do_settings_sections('coding-bunny-image-watermark');
						submit_button(esc_html__('Save Settings', 'coding-bunny-image-watermark'));
						?>
					</form>
					<hr>
					<p><b>If you feel you do not need the original watermark-free versions of your images, you can delete them to free up space on your server.<br>Keep in mind that this operation is irreversible.</b>
						<form method="post" onsubmit="return confirm('<?php esc_html_e('Are you sure you want to proceed? This action is irreversible.', 'coding-bunny-image-optimizer'); ?>');">
							<button type="submit" name="delete_backup_images" class="button"><?php esc_html_e('Delete All Backup Images', 'coding-bunny-image-optimizer'); ?></button>
						</form>
					</div>
				</div>
			</div>
			<p>
				© <?php echo esc_html(gmdate('Y')); ?> - 
				<?php esc_html_e('Powered by CodingBunny', 'coding-bunny-image-optimizer'); ?> |
				<a href="https://coding-bunny.com/support/" target="_blank" rel="noopener"><?php esc_html_e('Support', 'coding-bunny-image-optimizer'); ?></a> |
				<a href="https://coding-bunny.com/documentation/image-optimizer-doc/" target="_blank" rel="noopener"><?php esc_html_e('Documentation', 'coding-bunny-image-optimizer'); ?></a> |
				<a href="https://coding-bunny.com/changelog/" target="_blank" rel="noopener"><?php esc_html_e('Changelog', 'coding-bunny-image-optimizer'); ?></a>
			</p>
			<?php

			if (isset($_POST['delete_backup_images'])) {
				$this->delete_backup_images();
			}
		}

		public function delete_backup_images() {
			$upload_dir = wp_upload_dir();
			$backup_dir = $upload_dir['basedir'] . '/cbio_watermark_backups/';

			if (file_exists($backup_dir)) {
				$files = glob($backup_dir . '*');
				foreach ($files as $file) {
					wp_delete_file($file);
				}
			}
		}

		public function register_settings() {
			register_setting(
			'watermark_option_group',
			'watermark_options',
			array($this, 'sanitize_watermark_options')
		);

		add_settings_section(
		'setting_section_id',
		'',
		null,
		'coding-bunny-image-watermark'
	);

	add_settings_field(
	'enable_auto_watermark',
	'Enable Automatic Watermark',
	array($this, 'enable_auto_watermark_callback'),
	'coding-bunny-image-watermark',
	'setting_section_id'
);

add_settings_field(
'enable_bulk_actions',
'Enable Bulk Actions',
array($this, 'enable_bulk_actions_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);

add_settings_field(
'separator_1',
'',
array($this, 'separator_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);

add_settings_field(
'watermark_image',
'Watermark Image',
array($this, 'watermark_image_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);

add_settings_field(
'watermark_position',
'Watermark Position',
array($this, 'watermark_position_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);

add_settings_field(
'watermark_size',
'Watermark Size',
array($this, 'watermark_size_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);

add_settings_field(
'watermark_opacity',
'Watermark Opacity',
array($this, 'watermark_opacity_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);

add_settings_field(
'separator_2',
'',
array($this, 'separator_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);

add_settings_field(
'watermark_thumbnail_sizes',
'Watermark Thumbnail Sizes',
array($this, 'watermark_thumbnail_sizes_callback'),
'coding-bunny-image-watermark',
'setting_section_id'
);
}

public function sanitize_watermark_options($input) {
$sanitized_input = array();
foreach ($input as $key => $value) {
switch ($key) {
case 'enable_auto_watermark':
case 'enable_bulk_actions':
$sanitized_input[$key] = intval($value);
break;
case 'watermark_image':
$sanitized_input[$key] = esc_url_raw($value);
break;
case 'watermark_position':
$sanitized_input[$key] = sanitize_text_field($value);
break;
case 'watermark_size':
case 'watermark_opacity':
$sanitized_input[$key] = floatval($value);
break;
case 'watermark_thumbnail_sizes':
$sanitized_input[$key] = array_map('sanitize_text_field', $value);
break;
default:
$sanitized_input[$key] = sanitize_text_field($value);
break;
}
}
return $sanitized_input;
}

public function separator_callback() {
echo '<hr>';
}

public function watermark_thumbnail_sizes_callback() {
echo '<p>' . esc_html__('Select the thumbnail sizes to which you want to apply a watermark.', 'coding-bunny-image-optimizer') . '</p>';
echo '<br>';
$sizes = get_intermediate_image_sizes();
$selected_sizes = isset($this->options['watermark_thumbnail_sizes']) ? $this->options['watermark_thumbnail_sizes'] : array();
foreach ($sizes as $size) {
$checked = in_array($size, $selected_sizes) ? 'checked' : '';        
echo '<label class="cbio-toggle-label">
<input type="checkbox" class="cbio-toggle" name="watermark_options[watermark_thumbnail_sizes][]" value="' . esc_attr($size) . '" ' . esc_attr($checked) . '>
<span class="cbio-slider"></span>
' . esc_html($size) . '
</label><br>';
}
}

public function enable_auto_watermark_callback() {
$enable_auto_watermark = isset($this->options['enable_auto_watermark']) ? $this->options['enable_auto_watermark'] : '0';
?>
<label class="cbio-toggle-label">
<input type="checkbox" class="cbio-toggle" id="enable_auto_watermark" name="watermark_options[enable_auto_watermark]" value="1" <?php checked($enable_auto_watermark, '1'); ?> />
<span class="cbio-slider"></span>
<?php esc_html_e('Apply watermark when uploading new images.', 'coding-bunny-image-optimizer'); ?>
</label>
<?php
}

public function enable_bulk_actions_callback() {
$enable_bulk_actions = isset($this->options['enable_bulk_actions']) ? $this->options['enable_bulk_actions'] : '0';
?>
<label class="cbio-toggle-label">
<input type="checkbox" class="cbio-toggle" id="enable_bulk_actions" name="watermark_options[enable_bulk_actions]" value="1" <?php checked($enable_bulk_actions, '1'); ?> />
<span class="cbio-slider"></span>
<?php esc_html_e('Enable bulk actions in the library.', 'coding-bunny-image-optimizer'); ?>
</label>
<?php
}

public function register_bulk_actions($bulk_actions) {
$this->options = get_option('watermark_options');
if (isset($this->options['enable_bulk_actions']) && $this->options['enable_bulk_actions'] == '1') {
$bulk_actions['apply_watermark'] = 'Apply Watermark';
$bulk_actions['restore_images'] = 'Restore Original Images';
}
return $bulk_actions;
}

public function sanitize($input) {
$new_input = array();
if (isset($input['watermark_position'])) {
$new_input['watermark_position'] = sanitize_text_field($input['watermark_position']);
}
if (isset($input['watermark_size'])) {
$new_input['watermark_size'] = absint($input['watermark_size']);
}
if (isset($input['watermark_opacity'])) {
$new_input['watermark_opacity'] = absint($input['watermark_opacity']);
}
if (isset($input['watermark_image_id'])) {
$image_mime = get_post_mime_type($input['watermark_image_id']);
$allowed_mimes = array('image/jpeg', 'image/png', 'image/webp');

if (in_array($image_mime, $allowed_mimes)) {
$new_input['watermark_image_id'] = absint($input['watermark_image_id']);
} else {
add_settings_error(
'watermark_options',
'invalid_watermark_image',
__('Invalid watermark image type. Allowed types: JPEG, PNG and WebP.', 'coding-bunny-image-optimizer'),
'error'
);
}
}
if (isset($input['enable_auto_watermark'])) {
$new_input['enable_auto_watermark'] = $input['enable_auto_watermark'] ? '1' : '0';
}
if (isset($input['enable_bulk_actions'])) {
$new_input['enable_bulk_actions'] = $input['enable_bulk_actions'] ? '1' : '0';
}
if (isset($input['watermark_thumbnail_sizes'])) {
$new_input['watermark_thumbnail_sizes'] = array_map('sanitize_text_field', $input['watermark_thumbnail_sizes']);
}
return $new_input;
}

public function watermark_image_callback() {
echo '<p>' . esc_html__('Upload a PNG, JPEG or WebP image to use a watermark.', 'coding-bunny-image-optimizer') . '</p>';
echo '<br>';
$image_id = isset($this->options['watermark_image_id']) ? $this->options['watermark_image_id'] : '';
$image_url = $image_id ? wp_get_attachment_url($image_id) : '';

?>
<input type="hidden" name="watermark_options[watermark_image_id]" id="watermark_image_id" value="<?php echo esc_attr($image_id); ?>" />
<button type="button" class="button" id="choose-watermark-image"><?php esc_html_e('Choose Watermark Image', 'coding-bunny-image-optimizer'); ?></button>
<br>
<?php if ($image_url): ?>
<img src="<?php echo esc_url($image_url); ?>" id="watermark-image-preview" style="max-width: 150px; margin-top: 10px;" />
<?php endif; ?>
<?php
}

public function watermark_position_callback() {
$positions = array(
'top-left' => 'Top Left',
'top-center' => 'Top Center',
'top-right' => 'Top Right',
'center-left' => 'Center Left',
'center' => 'Center',
'center-right' => 'Center Right',
'bottom-left' => 'Bottom Left',
'bottom-center' => 'Bottom Center',
'bottom-right' => 'Bottom Right',
);
$selected_position = isset($this->options['watermark_position']) ? $this->options['watermark_position'] : 'bottom-right';
?>
<div class="watermark-position-radios">
<table>
<tr>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_top_left" name="watermark_options[watermark_position]" value="top-left" <?php checked($selected_position, 'top-left'); ?> />
<label for="position_top_left" class="cbio-watermark-label">Top Left</label>
</div>
</td>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_top_center" name="watermark_options[watermark_position]" value="top-center" <?php checked($selected_position, 'top-center'); ?> />
<label for="position_top_center" class="cbio-watermark-label">Top Center</label>
</div>
</td>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_top_right" name="watermark_options[watermark_position]" value="top-right" <?php checked($selected_position, 'top-right'); ?> />
<label for="position_top_right" class="cbio-watermark-label">Top Right</label>
</div>
</td>
</tr>
<tr>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_center_left" name="watermark_options[watermark_position]" value="center-left" <?php checked($selected_position, 'center-left'); ?> />
<label for="position_center_left" class="cbio-watermark-label">Center Left</label>
</div>
</td>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_center" name="watermark_options[watermark_position]" value="center" <?php checked($selected_position, 'center'); ?> />
<label for="position_center" class="cbio-watermark-label">Center</label>
</div>
</td>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_center_right" name="watermark_options[watermark_position]" value="center-right" <?php checked($selected_position, 'center-right'); ?> />
<label for="position_center_right" class="cbio-watermark-label">Center Right</label>
</div>
</td>
</tr>
<tr>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_bottom_left" name="watermark_options[watermark_position]" value="bottom-left" <?php checked($selected_position, 'bottom-left'); ?> />
<label for="position_bottom_left" class="cbio-watermark-label">Bottom Left</label>
</div>
</td>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_bottom_center" name="watermark_options[watermark_position]" value="bottom-center" <?php checked($selected_position, 'bottom-center'); ?> />
<label for="position_bottom_center" class="cbio-watermark-label">Bottom Center</label>
</div>
</td>
<td>
<div class="cbio-radio-button-wrapper">
<input type="radio" id="position_bottom_right" name="watermark_options[watermark_position]" value="bottom-right" <?php checked($selected_position, 'bottom-right'); ?> />
<label for="position_bottom_right" class="cbio-watermark-label">Bottom Right</label>
</div>
</td>
</tr>
</table>
</div>
<?php
}

public function watermark_size_callback() {
$size = isset($this->options['watermark_size']) ? $this->options['watermark_size'] : 20;
$watermark_image_id = isset($this->options['watermark_image_id']) ? $this->options['watermark_image_id'] : '';
$effective_dimensions = '';

if ($watermark_image_id) {
$image_url = wp_get_attachment_url($watermark_image_id);
if (!empty($image_url)) {
$image_info = getimagesize($image_url);
} else {
$image_info = null;
}
}
?>
<input type="range" name="watermark_options[watermark_size]" value="<?php echo esc_attr($size); ?>" min="1" max="100" oninput="this.nextElementSibling.value = this.value">
<output class="cbio-check-label"><?php echo esc_attr($size); ?></output>
<p><?php esc_html_e('Set a number between 0 and 100. 100 makes the width of the watermark image equal to the width of the image to which it is applied.', 'coding-bunny-image-optimizer'); ?></p>
<?php
}

public function watermark_opacity_callback() {
$opacity = isset($this->options['watermark_opacity']) ? $this->options['watermark_opacity'] : 100;
?>
<input type="range" name="watermark_options[watermark_opacity]" value="<?php echo esc_attr($opacity); ?>" min="0" max="100" oninput="this.nextElementSibling.value = this.value">
<output class="cbio-check-label"><?php echo esc_attr($opacity); ?></output>
<p><?php esc_html_e('Set a number between 0 and 100. 0 makes the watermark image completely transparent, while 100 displays the original image.', 'coding-bunny-image-optimizer'); ?></p>
<?php
}

public function enqueue_scripts() {
wp_enqueue_script('jquery');
wp_enqueue_media();
wp_enqueue_script('watermark-plugin-script', plugin_dir_url(__FILE__) . 'watermark.js', array('jquery'), '1.0', true);
}

public function apply_watermark_to_image($image_url, $is_thumbnail = false) {
if (empty($image_url)) {
return false;
}

$options = wp_parse_args(get_option('watermark_options'), [
'watermark_image_id' => null,
'watermark_size' => 50,
'watermark_opacity' => 50,
'watermark_padding' => 10,
'watermark_position' => 'bottom-right'
]);

$watermark_image = wp_get_attachment_image_src($options['watermark_image_id'], 'full');
if (!$watermark_image) {
return false;
}

try {
$watermark = Cbio_Watermark_Functions::create_image_resource($watermark_image[0]);
$original_image = Cbio_Watermark_Functions::create_image_resource($image_url);

if (!$watermark || !$original_image) {
throw new Exception('Failed to create image resources');
}

list($image_width, $image_height) = getimagesize($image_url);
list($watermark_width, $watermark_height) = getimagesize($watermark_image[0]);

$scale = $options['watermark_size'] / 100;
$new_watermark_width = (int)($image_width * $scale);
$new_watermark_height = (int)($new_watermark_width * $watermark_height / $watermark_width);

$watermark_resized = Cbio_Watermark_Functions::resize_image_with_transparency(
$watermark, 
$watermark_width, 
$watermark_height, 
$new_watermark_width, 
$new_watermark_height
);

Cbio_Watermark_Functions::apply_image_opacity($watermark_resized, $options['watermark_opacity']);

$position = Cbio_Watermark_Functions::calculate_watermark_position(
$image_width, 
$image_height, 
$new_watermark_width, 
$new_watermark_height, 
$options['watermark_position'], 
$options['watermark_padding']
);

imagecopy(
$original_image, 
$watermark_resized, 
(int)$position['x'], 
(int)$position['y'], 
0, 
0, 
(int)$new_watermark_width, 
(int)$new_watermark_height
);

$image_save_path = Cbio_Watermark_Functions::get_local_path_from_url($image_url);
Cbio_Watermark_Functions::save_image($original_image, $image_save_path);

return true;
} catch (Exception $e) {
return false;
} finally {
if (isset($watermark) && $watermark !== false) imagedestroy($watermark);
if (isset($original_image) && $original_image !== false) imagedestroy($original_image);
if (isset($watermark_resized) && $watermark_resized !== false) imagedestroy($watermark_resized); // Line 507
}
}

public function apply_watermark($metadata, $attachment_id) {
$file_url = wp_get_attachment_url($attachment_id);

$options = get_option('watermark_options');
if (isset($options['enable_auto_watermark']) && $options['enable_auto_watermark'] == '1') {
$upload_dir = wp_upload_dir();
$backup_dir = $upload_dir['basedir'] . '/cbio_watermark_backups/';

global $wp_filesystem;
if (empty($wp_filesystem)) {
require_once ABSPATH . 'wp-admin/includes/file.php';
WP_Filesystem();
}

if (!$wp_filesystem->is_dir($backup_dir)) {
$wp_filesystem->mkdir($backup_dir, 0755);
}

$backup_file_path = trailingslashit($backup_dir) . basename($file_url);
if (!$wp_filesystem->exists($backup_file_path)) {
if ($wp_filesystem->exists(Cbio_Watermark_Functions::get_local_path_from_url($file_url))) {
$wp_filesystem->copy(Cbio_Watermark_Functions::get_local_path_from_url($file_url), $backup_file_path);
}
}

if (isset($options['watermark_thumbnail_sizes'])) {
foreach ($options['watermark_thumbnail_sizes'] as $size) {
if (isset($metadata['sizes'][$size])) {
$thumbnail_file = $metadata['sizes'][$size]['file'];
$thumbnail_url = str_replace(basename($file_url), $thumbnail_file, $file_url);
$backup_thumbnail_path = trailingslashit($backup_dir) . basename($thumbnail_url);
if (!$wp_filesystem->exists($backup_thumbnail_path)) {
if ($wp_filesystem->exists(Cbio_Watermark_Functions::get_local_path_from_url($thumbnail_url))) {
$wp_filesystem->copy(Cbio_Watermark_Functions::get_local_path_from_url($thumbnail_url), $backup_thumbnail_path);
}
}
}
}
}

$this->apply_watermark_to_image($file_url);

if (isset($options['watermark_thumbnail_sizes'])) {
foreach ($options['watermark_thumbnail_sizes'] as $size) {
if (isset($metadata['sizes'][$size])) {
$thumbnail_file = $metadata['sizes'][$size]['file'];
$thumbnail_url = str_replace(basename($file_url), $thumbnail_file, $file_url);
$this->apply_watermark_to_image($thumbnail_url, true);
}
}
}
}

return $metadata;
}

public function handle_bulk_actions($redirect_to, $doaction, $attachment_ids) {

if (!is_array($attachment_ids) || empty($attachment_ids)) {
return $redirect_to;
}

try {
global $wp_filesystem;
if (empty($wp_filesystem)) {
require_once ABSPATH . 'wp-admin/includes/file.php';
WP_Filesystem();
}

$upload_dir = wp_upload_dir();
$backup_dir = trailingslashit($upload_dir['basedir']) . 'cbio_watermark_backups/';

if (!$wp_filesystem->is_dir($backup_dir)) {
if (!$wp_filesystem->mkdir($backup_dir, 0755)) {
return $redirect_to;
}
}

if ($doaction === 'apply_watermark') {
$processed_count = $this->process_watermark_batch($attachment_ids, $backup_dir);
$redirect_to = add_query_arg('bulk_watermarked', $processed_count, $redirect_to);
}

if ($doaction === 'restore_images') {
$restored_count = $this->restore_images_batch($attachment_ids, $backup_dir);
$redirect_to = add_query_arg('bulk_restored', $restored_count, $redirect_to);
}
} catch (Exception $e) {
}

return $redirect_to;
}

private function process_watermark_batch($attachment_ids, $backup_dir) {
$batch_size = 10;
$processed_count = 0;

foreach (array_chunk($attachment_ids, $batch_size) as $batch) {
foreach ($batch as $attachment_id) {
try {
$this->backup_and_watermark_image($attachment_id, $backup_dir);
$processed_count++;
} catch (Exception $e) {
}
}
}

return $processed_count;
}

private function restore_images_batch($attachment_ids, $backup_dir) {
$batch_size = 10;
$restored_count = 0;

foreach (array_chunk($attachment_ids, $batch_size) as $batch) {
foreach ($batch as $attachment_id) {
try {
$this->restore_single_image($attachment_id, $backup_dir);
$restored_count++;
} catch (Exception $e) {
}
}
}

return $restored_count;
}

private function backup_and_watermark_image($attachment_id, $backup_dir) {
$file_url = wp_get_attachment_url($attachment_id);
$backup_file_path = trailingslashit($backup_dir) . basename($file_url);

Cbio_Watermark_Functions::safe_backup_file($file_url, $backup_file_path);

$this->apply_watermark_to_image($file_url);

$metadata = wp_get_attachment_metadata($attachment_id);
if (isset($metadata['sizes'])) {
foreach ($metadata['sizes'] as $size) {
$thumbnail_file = $size['file'];
$thumbnail_url = str_replace(basename($file_url), $thumbnail_file, $file_url);

$backup_thumbnail_path = trailingslashit($backup_dir) . basename($thumbnail_url);
Cbio_Watermark_Functions::safe_backup_file($thumbnail_url, $backup_thumbnail_path);

$this->apply_watermark_to_image($thumbnail_url, true);
}
}
}

private function restore_single_image($attachment_id, $backup_dir) {
$file_url = wp_get_attachment_url($attachment_id);
$local_path = Cbio_Watermark_Functions::get_local_path_from_url($file_url);
$backup_file_path = $backup_dir . basename($local_path);

Cbio_Watermark_Functions::safe_restore_file($backup_file_path, $local_path);

$metadata = wp_get_attachment_metadata($attachment_id);
if (isset($metadata['sizes'])) {
foreach ($metadata['sizes'] as $size) {
$thumbnail_file = $size['file'];
$thumbnail_url = str_replace(basename($file_url), $thumbnail_file, $file_url);
$local_thumbnail_path = Cbio_Watermark_Functions::get_local_path_from_url($thumbnail_url);
$backup_thumbnail_path = $backup_dir . basename($local_thumbnail_path);

Cbio_Watermark_Functions::safe_restore_file($backup_thumbnail_path, $local_thumbnail_path);
}
}
}
}

if (is_admin()) {
$imageWatermarkPlugin = new Cbio_Watermark();
}
?>