<?php

if (!defined('ABSPATH')) {
	exit;
}

function cbio_schedule_license_check() {
	if (!wp_next_scheduled('cbio_check_license')) {
		wp_schedule_event(time(), 'daily', 'cbio_check_license');
	}
}
add_action('wp', 'cbio_schedule_license_check');

function cbio_do_license_check() {
	$licence_data = get_option('cbio_licence_data', ['key' => '', 'email' => '']);
	if (!empty($licence_data['key']) && !empty($licence_data['email'])) {
		delete_transient('cbio_licence_validation_' . md5($licence_data['key'] . $licence_data['email']));
		cbio_validate_licence($licence_data['key'], $licence_data['email']);
	}
}
add_action('cbio_check_license', 'cbio_do_license_check');

function cbio_submenu() {
	add_submenu_page(
	'coding-bunny-image-optimizer',
	esc_html__("Manage License", 'coding-bunny-image-optimizer'),
	esc_html__("Manage License", 'coding-bunny-image-optimizer'),
	'manage_options',
	'cbio-licence',
	'cbio_licence_page'
);

$licence_data = get_option('cbio_licence_data', ['key' => '', 'email' => '']);
$licence_key = sanitize_text_field($licence_data['key']);
$licence_email = sanitize_email($licence_data['email']);
$licence_active = cbio_validate_licence($licence_key, $licence_email);
}
add_action('admin_menu', 'cbio_submenu');

function cbio_licence_page() {
global $wpdb;
$licence_data = get_option('cbio_licence_data', ['key' => '', 'email' => '']);
$licence_key = $licence_data['key'];
$licence_email = $licence_data['email'];
$domain = wp_parse_url(get_site_url(), PHP_URL_HOST);

if (isset($_GET['refresh'])) {
	delete_transient('cbio_licence_validation_' . md5($licence_key . $licence_email));
	delete_transient('cbio_domain_status_' . md5($licence_key . $domain));
}

$licence_active = cbio_validate_licence($licence_key, $licence_email);

$domain_status = ['is_active' => false, 'admin_deactivated' => false, 'message' => ''];
if ($licence_active['success']) {
	$domain_status = cbio_check_domain_status($licence_key, $licence_email, $domain);

	if (!$domain_status['is_active']) {
		$licence_active['success'] = false;
		$licence_active['domain_inactive'] = true;
		$licence_active['error'] = $domain_status['message'];
	}
}

if (isset($_POST['validate_licence']) && check_admin_referer('cbio_licence_validation')) {
	$licence_key = isset($_POST['licence_key']) ? sanitize_text_field(wp_unslash($_POST['licence_key'])) : '';
	$licence_email = isset($_POST['licence_email']) ? sanitize_email(wp_unslash($_POST['licence_email'])) : '';
	$response = cbio_validate_licence($licence_key, $licence_email);

	if ($response['success']) {
		$domain_check = cbio_check_domain_status($licence_key, $licence_email, $domain);
		if (!$domain_check['is_active']) {
			$response['success'] = false;
			$response['domain_inactive'] = true;
			$response['error'] = $domain_check['message'];
		}
	}

	if ($response['success']) {
		update_option('cbio_licence_data', ['key' => $licence_key, 'email' => $licence_email]);
		echo '<div class="notice notice-success"><p>' . esc_html__("License successfully validated!", 'coding-bunny-image-optimizer') . '</p></div>';
		echo '<script>setTimeout(function(){ location.reload(); }, 1000);</script>';
	} else {
		$error_message = isset($response['domain_inactive']) 
			? esc_html__("Domain is inactive: ", 'coding-bunny-image-optimizer') 
				: esc_html__("Incorrect license key or email: ", 'coding-bunny-image-optimizer');

		echo '<div class="notice notice-error"><p>' . esc_html($error_message) . ' ' . esc_html($response['error']) . '</p></div>';
	}
}

if (isset($_POST['deactivate_licence']) && check_admin_referer('cbio_licence_deactivation')) {
	$licence_key = $licence_data['key'];
	$licence_email = $licence_data['email'];

	if (!empty($licence_key) && !empty($licence_email)) {
		$domain = wp_parse_url(get_site_url(), PHP_URL_HOST);
		$url = esc_url_raw('https://coding-bunny.com/plugin-license-new/io-active-licence.php');

		$response = wp_remote_post($url, [
			'body' => wp_json_encode([
				'licence_key' => $licence_key,
				'email' => $licence_email,
				'domain' => $domain,
				'action' => 'deactivate'
				]),
				'headers' => [
				'Content-Type' => 'application/json',
			],
			'timeout' => 15,
			'sslverify' => true,
			]);

			delete_transient('cbio_licence_validation_' . md5($licence_key . $licence_email));
			delete_transient('cbio_domain_status_' . md5($licence_key . $domain));
		}

		delete_option('cbio_licence_data');
		$licence_key = '';
		$licence_email = '';
		echo '<div class="notice notice-success"><p>' . esc_html__("License successfully deactivated!", 'coding-bunny-image-optimizer') . '</p></div>';
		echo '<script>setTimeout(function(){ location.reload(); }, 1000);</script>';
	}

	?>
	<div class="wrap">
		<h1><?php esc_html_e('CodingBunny Image Optimizer PRO', 'coding-bunny-image-optimizer'); ?> 
			<span>v<?php echo esc_html(CBIO_VERSION); ?></span></h1>
				<div class="cbio-section">
					<h3><?php esc_html_e('Manage License', 'coding-bunny-image-optimizer'); ?></h3>
					<p>
						<?php esc_html_e('Enter your registration email and license key.', 'coding-bunny-image-optimizer'); ?>
					</p>
				<form method="post" action="">
					<?php wp_nonce_field('cbio_licence_validation'); ?>
					<div class="cbio-flex-container">
						<div class="cbio-flex-item">
							<label for="licence_email"><?php esc_html_e("Email account:", 'coding-bunny-image-optimizer'); ?></label>
						</div>
						<div class="cbio-flex-item">
						    <input type="email" id="licence_email" name="licence_email" 
						    value="<?php echo esc_attr($licence_active['success'] ? str_repeat('*', strlen($licence_email)) : $licence_email); ?>" 
						    required />
						</div>
						<div class="cbio-flex-item">
							<label for="licence_key"><?php esc_html_e("License Key:", 'coding-bunny-image-optimizer'); ?></label>
						</div>
						<div class="cbio-flex-item">
							<input type="text" id="licence_key" name="licence_key" 
							value="<?php echo esc_attr($licence_active['success'] ? str_repeat('*', strlen($licence_key)) : $licence_key); ?>" 
							required />   
						</div>
						<div class="cbio-flex-item">
							<?php if ($licence_active['success']) : ?>
								<?php if ($domain_status['is_active'] || !$domain_status['admin_deactivated']) : ?>
									<?php wp_nonce_field('cbio_licence_deactivation'); ?>
									<button type="submit" name="deactivate_licence" class="button button-primary">
										<?php esc_html_e("Deactivate license", 'coding-bunny-image-optimizer'); ?>
									</button>
								<?php else : ?>
									<div class="notice notice-error inline">
										<p><?php esc_html_e("This domain has been deactivated by an administrator. Please contact support for assistance.", 'coding-bunny-image-optimizer'); ?></p>
									</div>
								<?php endif; ?>
							<?php else : ?>
								<button type="submit" name="validate_licence" class="button button-primary">
									<?php esc_html_e("Activate license", 'coding-bunny-image-optimizer'); ?>
								</button>
							<?php endif; ?>
						</div>
					</div>
					<?php if ($licence_active['success']) : ?>
						<div style="margin-top: 20px;">        
							<div style="margin-top: 20px;">
								<?php esc_html_e("Status license:", 'coding-bunny-image-optimizer'); ?>
								<span class="cbio-php-module-label cbio-enabled-lib">Active</span>
							</div>

							<div style="margin-top: 10px;">
								<?php esc_html_e("Your license expires on:", 'coding-bunny-image-optimizer'); ?>
								<span style="font-weight: bold;">
									<?php 
									$expiration_date = DateTime::createFromFormat('Y-m-d', $licence_active['expiration']);
									if ($expiration_date->format('Y') >= 2099) {
										echo esc_html__("LIFETIME", 'coding-bunny-image-optimizer');
									} else {
										echo esc_html($expiration_date->format('d-m-Y'));
									}
									?>
								</span>
							</div>

						</div>
					<?php else : ?>
						<div style="margin-top: 20px;">
							<div style="margin-top: 20px;">
								<?php esc_html_e("Status license:", 'coding-bunny-image-optimizer'); ?>
								<span class="cbio-php-module-label cbio-disabled-lib">Inactive</span>
							</div>

						</div>
					<?php endif; ?>

					<div style="margin-top: 15px;">
						<a href="<?php echo esc_url(add_query_arg('refresh', 'true')); ?>" class="button button-primary">
							<?php esc_html_e('Refresh License Status', 'coding-bunny-image-optimizer'); ?>
						</a>
					</div>
				</form>
			</div>
	        <p>
	            © <?php echo esc_html( gmdate( 'Y' ) ); ?> - 
	            <?php esc_html_e( 'Powered by CodingBunny', 'coding-bunny-image-optimizer' ); ?> |
	            <a href="https://coding-bunny.com/support/" target="_blank" rel="noopener"><?php esc_html_e( 'Support', 'coding-bunny-image-optimizer' ); ?></a> |
	            <a href="https://coding-bunny.com/documentation/image-optimizer-doc/" target="_blank" rel="noopener"><?php esc_html_e( 'Documentation', 'coding-bunny-image-optimizer' ); ?></a> |
	            <a href="https://coding-bunny.com/changelog/" target="_blank" rel="noopener"><?php esc_html_e( 'Changelog', 'coding-bunny-image-optimizer' ); ?></a>
	        </p>
		</div>
		<style>
			.cbio-php-module-label {
				display: inline-block;
				padding: 3px 8px;
				border-radius: 3px;
				font-weight: bold;
			}
			.cbio-enabled-lib {
				background-color: #dff0d8;
				color: #3c763d;
			}
			.cbio-disabled-lib {
				background-color: #f2dede;
				color: #a94442;
			}
			.notice.inline {
				margin: 5px 0;
				padding: 5px 10px;
			}
			</style>
			<?php 
		}

		function cbio_check_domain_status($licence_key, $licence_email, $domain) {
			$cache_key = 'cbio_domain_status_' . md5($licence_key . $domain);
			$cache_ttl = 1 * HOUR_IN_SECONDS;

			$cached_status = get_transient($cache_key);
			if ($cached_status !== false) {
				return $cached_status;
			}

			$url = esc_url_raw('https://coding-bunny.com/plugin-license-new/io-active-licence.php');

			$response = wp_remote_post($url, [
				'body' => wp_json_encode([
					'licence_key' => $licence_key,
					'email' => $licence_email,
					'domain' => $domain,
					'action' => 'check_domain'
					]),
					'headers' => [
					'Content-Type' => 'application/json',
				],
				'timeout' => 15,
				'sslverify' => true,
				]);

				if (is_wp_error($response)) {
					$status = ['is_active' => false, 'admin_deactivated' => false, 'message' => $response->get_error_message()];
					set_transient($cache_key, $status, 5 * MINUTE_IN_SECONDS);
					return $status;
				}

				$body = json_decode(wp_remote_retrieve_body($response), true);

				if (isset($body['success']) && $body['success']) {
					$admin_deactivated = isset($body['admin_deactivated']) ? $body['admin_deactivated'] : false;

					if (isset($body['domain_active']) && $body['domain_active']) {
						$status = ['is_active' => true, 'admin_deactivated' => false, 'message' => ''];
					} else {
						$message = isset($body['message']) ? $body['message'] : 'This domain is not active for this license';
						$status = [
							'is_active' => false, 
							'admin_deactivated' => $admin_deactivated, 
							'message' => $message
						];
					}

					set_transient($cache_key, $status, $cache_ttl);
					return $status;
				} else {
					$error = isset($body['error']) ? $body['error'] : 'Error during domain verification';
					$status = ['is_active' => false, 'admin_deactivated' => false, 'message' => $error];

					set_transient($cache_key, $status, 5 * MINUTE_IN_SECONDS);
					return $status;
				}
			}

			function cbio_validate_licence($licence_key, $licence_email) {
				if (empty($licence_key) || empty($licence_email)) {
					return ['success' => false, 'error' => 'Missing license key or email'];
				}

				$cache_key = 'cbio_licence_validation_' . md5($licence_key . $licence_email);
				$cache_ttl = 4 * HOUR_IN_SECONDS;

				$licence_active = get_transient($cache_key);
				if ($licence_active !== false) {
					$domain = wp_parse_url(get_site_url(), PHP_URL_HOST);
					$domain_status = cbio_check_domain_status($licence_key, $licence_email, $domain);

					if (!$domain_status['is_active']) {
						return [
							'success' => false,
							'domain_inactive' => true,
							'error' => $domain_status['message']
						];
					}

					return $licence_active;
				}

				$url = esc_url_raw('https://coding-bunny.com/plugin-license-new/io-active-licence.php');

				$domain = wp_parse_url(get_site_url(), PHP_URL_HOST);

				$response = wp_remote_post($url, [
					'body' => wp_json_encode([
						'licence_key' => sanitize_text_field($licence_key),
						'email' => sanitize_email($licence_email),
						'domain' => $domain,
						'action' => 'validate'
						]),
						'headers' => [
						'Content-Type' => 'application/json',
					],
					'timeout' => 15,
					'sslverify' => true,
					]);

					if (is_wp_error($response)) {
						return ['success' => false, 'error' => $response->get_error_message()];
					}

					$body = json_decode(wp_remote_retrieve_body($response), true);

					if (isset($body['success']) && $body['success']) {
						$domain_status = cbio_check_domain_status($licence_key, $licence_email, $domain);

						if (!$domain_status['is_active']) {
							return [
								'success' => false,
								'domain_inactive' => true,
								'error' => $domain_status['message']
							];
						}

						$licence_active = [
							'success' => true,
							'expiration' => sanitize_text_field($body['expiration']),
							'activations' => isset($body['activations']) ? $body['activations'] : []
						];

						set_transient($cache_key, $licence_active, $cache_ttl);
						return $licence_active;
					} else {
						$licence_active = [
							'success' => false,
							'error' => isset($body['error']) ? sanitize_text_field($body['error']) : esc_html__("Incorrect licence key or email", 'coding-bunny-image-optimizer')
						];

						set_transient($cache_key, $licence_active, 30 * MINUTE_IN_SECONDS);
						return $licence_active;
					}
				}

				function cbio_is_licence_valid_and_active() {
					$licence_data = get_option('cbio_licence_data', ['key' => '', 'email' => '']);
					$licence_key = sanitize_text_field($licence_data['key']);
					$licence_email = sanitize_email($licence_data['email']);

					if (empty($licence_key) || empty($licence_email)) {
						return false;
					}

					$licence_status = cbio_validate_licence($licence_key, $licence_email);

					return isset($licence_status['success']) && $licence_status['success'] === true;
				}