<?php

if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

function cbio_resize_image( $file ) {

    static $options_cache = null;
    static $image_extensions = array( 'jpg', 'jpeg', 'png', 'gif', 'webp' );
    
    if ( $options_cache === null ) {
        $options_cache = array(
            'enable_resize' => get_option( 'cbio_enable_resize', '1' ),
            'max_width'     => (int) get_option( 'cbio_max_width', 1000 ),
            'max_height'    => (int) get_option( 'cbio_max_height', 1000 ),
        );
        
        $options_cache = apply_filters( 'cbio_resize_options_cache', $options_cache );
    }
    
    if ( $options_cache['enable_resize'] !== '1' ) {
        return $file;
    }

    if ( !file_exists( $file['file'] ) ) {
        return $file;
    }

    $file_ext = strtolower( pathinfo( $file['file'], PATHINFO_EXTENSION ) );
    
    if ( !in_array( $file_ext, $image_extensions, true ) ) {
        return $file;
    }
    
    static $file_info_cache = array();
    $file_path = $file['file'];
    $file_hash = md5($file_path);
    
    if ( !isset( $file_info_cache[$file_hash] ) ) {

        $mime_type = mime_content_type( $file_path );
        
        if ( strpos( $mime_type, 'image' ) === false ) {

            $file_info_cache[$file_hash] = false;
            return $file;
        }
        
        $image_size = @getimagesize( $file_path );
        
        if ( !$image_size ) {
            cbio_add_admin_notice( 
                sprintf(
					// translators: %s is replaced with the file URL that couldn't be processed
                    esc_html__('Error: Unable to get image dimensions for file %s', 'coding-bunny-image-optimizer'), 
                    esc_url( $file_path ) 
                ), 
                'error' 
            );

            $file_info_cache[$file_hash] = false;
            return $file;
        }
        
        $file_info_cache[$file_hash] = array(
            'mime_type' => $mime_type,
            'width'     => $image_size[0],
            'height'    => $image_size[1],
        );
        
        if ( count($file_info_cache) > 50 ) {
            array_shift( $file_info_cache );
        }
    } elseif ( $file_info_cache[$file_hash] === false ) {
        return $file;
    }
    
    $image_info = $file_info_cache[$file_hash];
    
    if ( $image_info['width'] <= $options_cache['max_width'] && $image_info['height'] <= $options_cache['max_height'] ) {
        return $file;
    }

    $original_width = $image_info['width'];
    $original_height = $image_info['height'];
    $ratio = $original_width / $original_height;
    
    $new_width = $original_width;
    $new_height = $original_height;
    
    if ( $original_width > $options_cache['max_width'] ) {
        $new_width = $options_cache['max_width'];
        $new_height = $new_width / $ratio;
    }
    
    if ( $new_height > $options_cache['max_height'] ) {
        $new_height = $options_cache['max_height'];
        $new_width = $new_height * $ratio;
    }
    
    if ( abs($new_width - $original_width) < 10 && abs($new_height - $original_height) < 10 ) {
        return $file;
    }
    
    $image_editor = wp_get_image_editor( $file_path );
    if ( is_wp_error( $image_editor ) ) {
        cbio_add_admin_notice( 
            sprintf(
				// translators: %s is replaced with the file URL that couldn't be processed
                esc_html__('Error: Unable to create image editor for file %s', 'coding-bunny-image-optimizer'), 
                esc_url( $file_path ) 
            ), 
            'error' 
        );
        return $file;
    }
    
    $result = $image_editor->resize( (int) $new_width, (int) $new_height, false );
    if ( is_wp_error( $result ) ) {
        cbio_add_admin_notice( 
            sprintf(
				// translators: %s is replaced with the file URL that couldn't be processed
                esc_html__('Error: Unable to resize image for file %s', 'coding-bunny-image-optimizer'), 
                esc_url( $file_path ) 
            ), 
            'error' 
        );
        return $file;
    }

    $saved = $image_editor->save( $file_path );
    if ( is_wp_error( $saved ) ) {
        cbio_add_admin_notice( 
            sprintf(
				// translators: %s is replaced with the file URL that couldn't be processed
                esc_html__('Error: Unable to save resized image for file %s', 'coding-bunny-image-optimizer'), 
                esc_url( $file_path ) 
            ), 
            'error' 
        );
    } else {

        $file_info_cache[$file_hash] = array(
            'mime_type' => $image_info['mime_type'],
            'width'     => (int) $new_width,
            'height'    => (int) $new_height,
        );
    }

    return $file;
}

function cbio_add_admin_notice( $message, $type = 'error' ) {
    static $notices = array();
    $message_hash = md5($message);
    
    if ( in_array( $message_hash, $notices ) ) {
        return;
    }
    
    $notices[] = $message_hash;
    
    add_action( 'admin_notices', function() use ( $message, $type ) {
        printf( 
            '<div class="notice notice-%s is-dismissible"><p>%s</p></div>',
            esc_attr( $type ),
            wp_kses_post( $message )
        );
    });
}

add_filter( 'wp_handle_upload', 'cbio_resize_image' );

?>