<?php

if (!defined('ABSPATH')) {
    exit;
}

function cbio_get_image_stats() {
    $args = array(
        'post_type'      => 'attachment',
        'post_mime_type' => 'image',
        'post_status'    => 'inherit',
        'posts_per_page' => -1,
    );

    $query = new WP_Query($args);

    $total_size = 0;
    $optimized_count = 0;
    $image_counts = array(
        'jpeg' => 0,
        'png'  => 0,
        'gif'  => 0,
        'webp' => 0,
        'avif' => 0,
        'svg'  => 0,
    );

    foreach ($query->posts as $attachment) {
        $file_path = get_attached_file($attachment->ID);

        if (file_exists($file_path)) {

            $file_size = filesize($file_path);
            $total_size += $file_size;

            $file_info = wp_check_filetype($file_path);
            $mime_type = $file_info['type'];
            
            cbio_process_image_mime_type($mime_type, $image_counts);

            if (in_array($mime_type, ['image/webp', 'image/avif', 'image/svg+xml'])) {
                $optimized_count++;
            }

            $meta_data = wp_get_attachment_metadata($attachment->ID);
            if (isset($meta_data['sizes']) && is_array($meta_data['sizes'])) {
                foreach ($meta_data['sizes'] as $size) {
                    $intermediate_file_path = path_join(dirname($file_path), $size['file']);
                    if (file_exists($intermediate_file_path)) {
                        $intermediate_file_size = filesize($intermediate_file_path);
                        $total_size += $intermediate_file_size;

                        $intermediate_file_info = wp_check_filetype($intermediate_file_path);
                        $intermediate_mime_type = $intermediate_file_info['type'];
                        
                        cbio_process_image_mime_type($intermediate_mime_type, $image_counts);
                        
                        if (in_array($intermediate_mime_type, ['image/webp', 'image/avif', 'image/svg+xml'])) {
                            $optimized_count++;
                        }
                    }
                }
            }
        }
    }

    $total_count = array_sum($image_counts);
    $optimized_percentage = ($total_count > 0) ? ($optimized_count / $total_count) * 100 : 0;

    return array(
        'total_count' => $total_count,
        'total_size'  => $total_size / (1024 * 1024), // Convert to MB
        'image_counts' => $image_counts,
        'optimized_percentage' => $optimized_percentage,
    );
}

function cbio_process_image_mime_type($mime_type, &$image_counts) {
    if (empty($mime_type) || strpos($mime_type, 'image/') !== 0) {
        return false;
    }
    
    $format_map = [
        'image/jpeg' => 'jpeg',
        'image/png' => 'png',
        'image/gif' => 'gif',
        'image/webp' => 'webp',
        'image/avif' => 'avif',
        'image/svg+xml' => 'svg'
    ];
    
    if (isset($format_map[$mime_type])) {
        $image_counts[$format_map[$mime_type]]++;
        return true;
    }
    
    return false;
}

function cbio_get_cached_image_stats() {
    $stats = get_transient('cbio_image_stats');

    if (false === $stats) {
        $stats = cbio_get_image_stats();
        set_transient('cbio_image_stats', $stats, 12 * HOUR_IN_SECONDS);
    }

    return $stats;
}

function cbio_update_image_stats() {
    $stats = cbio_get_image_stats();
    set_transient('cbio_image_stats', $stats, 12 * HOUR_IN_SECONDS);
}

function cbio_incremental_update_image_stats($attachment_id) {
    cbio_update_image_stats();
}

if (!wp_next_scheduled('cbio_update_image_stats')) {
    wp_schedule_event(time(), 'twicedaily', 'cbio_update_image_stats');
}

add_action('cbio_update_image_stats', 'cbio_update_image_stats');
add_action('add_attachment', 'cbio_incremental_update_image_stats');