<?php

if (!defined('ABSPATH')) {
	exit;
}

class Cbio_Watermark_Functions {

    public static function create_image_resource($path) {
        $image_info = getimagesize($path);

        switch ($image_info[2]) {
            case IMAGETYPE_JPEG:
                return imagecreatefromjpeg($path);
            case IMAGETYPE_PNG:
                $image = imagecreatefrompng($path);
                imagealphablending($image, false);
                imagesavealpha($image, true);
                return $image;
            case IMAGETYPE_GIF:
                return imagecreatefromgif($path);
            case IMAGETYPE_WEBP:
                return imagecreatefromwebp($path);
            default:
                return false;
        }
    }

    public static function resize_image_with_transparency($source_image, $src_width, $src_height, $dest_width, $dest_height) {
        $resized_image = imagecreatetruecolor($dest_width, $dest_height);
        imagealphablending($resized_image, false);
        imagesavealpha($resized_image, true);

        imagecopyresampled(
            $resized_image, 
            $source_image, 
            0, 0, 0, 0, 
            $dest_width, 
            $dest_height, 
            $src_width, 
            $src_height
        );

        return $resized_image;
    }

    public static function apply_image_opacity(&$image, $opacity_percentage) {
        $opacity = $opacity_percentage / 100;
        imagefilter($image, IMG_FILTER_COLORIZE, 0, 0, 0, intval(127 * (1 - $opacity)));
    }

    public static function calculate_watermark_position(
        $image_width, 
        $image_height, 
        $watermark_width, 
        $watermark_height, 
        $position, 
        $padding
    ) {
        $positions = [
            'top-left' => ['x' => $padding, 'y' => $padding],
            'top-center' => ['x' => ($image_width / 2) - ($watermark_width / 2), 'y' => $padding],
            'top-right' => ['x' => $image_width - $watermark_width - $padding, 'y' => $padding],
            'center-left' => ['x' => $padding, 'y' => ($image_height / 2) - ($watermark_height / 2)],
            'center' => ['x' => ($image_width / 2) - ($watermark_width / 2), 'y' => ($image_height / 2) - ($watermark_height / 2)],
            'center-right' => ['x' => $image_width - $watermark_width - $padding, 'y' => ($image_height / 2) - ($watermark_height / 2)],
            'bottom-left' => ['x' => $padding, 'y' => $image_height - $watermark_height - $padding],
            'bottom-center' => ['x' => ($image_width / 2) - ($watermark_width / 2), 'y' => $image_height - $watermark_height - $padding],
            'bottom-right' => ['x' => $image_width - $watermark_width - $padding, 'y' => $image_height - $watermark_height - $padding]
        ];

        return $positions[$position] ?? $positions['bottom-right'];
    }

    public static function save_image($image, $path) {
        $image_info = getimagesize($path);

        switch ($image_info[2]) {
            case IMAGETYPE_JPEG:
                imagejpeg($image, $path, 100);
                break;
            case IMAGETYPE_PNG:
                imagepng($image, $path);
                break;
            case IMAGETYPE_GIF:
                imagegif($image, $path);
                break;
            case IMAGETYPE_WEBP:
                imagewebp($image, $path);
                break;
            default:
                throw new Exception('Unsupported image type for saving');
        }
    }

    public static function get_local_path_from_url($url) {
        $upload_dir = wp_upload_dir();
        $relative_path = str_replace($upload_dir['baseurl'], '', $url);
        return $upload_dir['basedir'] . $relative_path;
    }

    public static function safe_backup_file($source_url, $backup_path) {
    global $wp_filesystem;

    if (!$wp_filesystem->exists($backup_path)) {
        $source_path = self::get_local_path_from_url($source_url);
        if ($wp_filesystem->exists($source_path)) {
            if (!$wp_filesystem->copy($source_path, $backup_path)) {
                throw new Exception(sprintf("Unable to back up %s", esc_html($source_url)));
            }
        }
    }
}

    public static function safe_restore_file($backup_path, $destination_path) {
    if (file_exists($backup_path)) {
        if (copy($backup_path, $destination_path)) {
            wp_delete_file($backup_path);
        } else {
            throw new Exception(sprintf("Unable to restore file from %s", esc_html($backup_path)));
        }
    }
}

    public static function get_cached_watermark($image_width, $opacity, $size) {
        if (!is_numeric($image_width) || !is_numeric($opacity) || !is_numeric($size)) {
            throw new InvalidArgumentException('Invalid parameters for watermarking');
        }

        $cache_key = sprintf(
            'watermark_%d_%d_%d', 
            (int)$image_width, 
            (int)$opacity, 
            (int)$size
        );

        $cache_group = 'image_watermarks';
        $cached_watermark = wp_cache_get($cache_key, $cache_group);

        if ($cached_watermark === false) {
            try {
                $cached_watermark = self::generate_watermark($image_width, $opacity, $size);
                $cache_expiration = apply_filters(
                    'watermark_cache_expiration', 
                    3600
                );

                wp_cache_set($cache_key, $cached_watermark, $cache_group, $cache_expiration);
            } catch (Exception $e) {
                return null;
            }
        }

        return $cached_watermark;
    }
}